import 'dart:io';

import 'package:admin_desktop/presentation/components/custom_line.dart';
import 'package:admin_desktop/presentation/pages/main/widgets/order_detail/print_page.dart';
import 'package:flutter/material.dart';
import 'package:flutter_esc_pos_utils/flutter_esc_pos_utils.dart';
import 'package:flutter_pos_printer_platform_image_3/flutter_pos_printer_platform_image_3.dart';
import 'package:flutter_screenutil/flutter_screenutil.dart';
import 'package:admin_desktop/domain/models/models.dart';
import 'package:admin_desktop/infrastructure/services/utils.dart';
import 'package:admin_desktop/presentation/components/components.dart';
import 'package:remixicon_updated/remixicon_updated.dart';

import 'package:admin_desktop/presentation/components/buttons/custom_mini_button.dart';
import 'package:admin_desktop/presentation/styles/style.dart';

class GenerateBookingData extends StatefulWidget {
  final BookingData? bookingData;

  const GenerateBookingData({super.key, required this.bookingData});

  @override
  State<GenerateBookingData> createState() => _GenerateBookingDataState();
}

class _GenerateBookingDataState extends State<GenerateBookingData> {
  @override
  Widget build(BuildContext context) {
    return Container(
      decoration: BoxDecoration(
        color: Style.white,
        borderRadius: BorderRadius.circular(10),
      ),
      padding: REdgeInsets.symmetric(horizontal: 14, vertical: 16),
      child: SingleChildScrollView(
        child: Column(
          mainAxisSize: MainAxisSize.min,
          crossAxisAlignment: CrossAxisAlignment.center,
          children: [
            Text(AppHelpers.getTranslation(TrKeys.bookingInvoice),
                style: Style.interMedium(size: 20)),
            8.verticalSpace,
            Text(
                "${AppHelpers.getTranslation(TrKeys.booking)} #${AppHelpers.getTranslation(TrKeys.id)}${widget.bookingData?.id}",
                style: Style.interNormal(size: 16)),
            12.verticalSpace,
            CustomLine(),
            12.verticalSpace,
            Row(
              children: [
                Text(AppHelpers.getTranslation(TrKeys.shopName),
                    style: Style.interRegular(size: 14)),
                Flexible(
                  child: Text(
                    LocalStorage.getShop()?.translation?.title ?? "",
                    style: Style.interRegular(size: 14),
                    textAlign: TextAlign.end,
                  ),
                )
              ],
            ),
            8.verticalSpace,
            Row(
              children: [
                Text(AppHelpers.getTranslation(TrKeys.client),
                    style: Style.interRegular(size: 14)),
                Spacer(),
                Text(
                    "${widget.bookingData?.user?.firstname ?? ""} ${widget.bookingData?.user?.lastname ?? ""}",
                    style: Style.interRegular(size: 14))
              ],
            ),
            8.verticalSpace,
            Row(
              children: [
                Text(AppHelpers.getTranslation(TrKeys.master),
                    style: Style.interRegular(size: 14)),
                Spacer(),
                Text(
                    "${widget.bookingData?.master?.firstname ?? ""} ${widget.bookingData?.master?.lastname ?? ""}",
                    style: Style.interRegular(size: 14))
              ],
            ),
            8.verticalSpace,
            Row(
              children: [
                Text(AppHelpers.getTranslation(TrKeys.date),
                    style: Style.interRegular(size: 14)),
                Spacer(),
                Text(TimeService.dateFormatMDYHm(widget.bookingData?.createdAt),
                    style: Style.interRegular(size: 12))
              ],
            ),
            10.verticalSpace,
            const Divider(thickness: 2),
            10.verticalSpace,
            CustomLine(),
            20.verticalSpace,
            Row(
              children: [
                Text(
                  AppHelpers.getTranslation(TrKeys.service),
                  style: Style.interRegular(size: 14),
                ),
                Spacer(),
                Text(
                  widget.bookingData?.serviceMaster?.service?.translation
                          ?.title ??
                      "",
                  style: Style.interRegular(size: 14),
                )
              ],
            ),
            10.verticalSpace,
            Row(
              children: [
                Text(
                  AppHelpers.getTranslation(TrKeys.bookingDate),
                  style: Style.interRegular(size: 14),
                ),
                Spacer(),
                Text(
                  TimeService.dateFormatDMY(widget.bookingData?.startDate),
                  style: Style.interRegular(size: 14),
                )
              ],
            ),
            8.verticalSpace,
            Row(
              children: [
                Text(
                  AppHelpers.getTranslation(TrKeys.bookingTime),
                  style: Style.interRegular(size: 14),
                ),
                Spacer(),
                Text(
                    "${TimeService.timeFormat(widget.bookingData?.startDate)}-${TimeService.timeFormat(widget.bookingData?.endDate)}",
                    style: Style.interRegular(size: 14))
              ],
            ),
            16.verticalSpace,
            CustomLine(),
            20.verticalSpace,
            Row(
              children: [
                Text(
                  AppHelpers.getTranslation(TrKeys.subtotal),
                  style: Style.interNormal(size: 14),
                ),
                const Spacer(),
                Text(
                  AppHelpers.numberFormat(number: widget.bookingData?.price),
                  style: Style.interRegular(size: 14),
                )
              ],
            ),
            if ((widget.bookingData?.discount ?? 0) != 0)
              Column(
                children: [
                  10.verticalSpace,
                  Row(
                    children: [
                      Text(
                        AppHelpers.getTranslation(TrKeys.discount),
                        style: Style.interNormal(size: 14),
                      ),
                      const Spacer(),
                      Text(
                        "-${AppHelpers.numberFormat(number: widget.bookingData?.discount)}",
                        style: Style.interRegular(size: 14),
                      )
                    ],
                  ),
                ],
              ),
            if ((widget.bookingData?.couponPrice ?? 0) != 0)
              Column(
                children: [
                  10.verticalSpace,
                  Row(
                    children: [
                      Text(
                        AppHelpers.getTranslation(TrKeys.coupon),
                        style: Style.interNormal(size: 14),
                      ),
                      const Spacer(),
                      Text(
                          "-${AppHelpers.numberFormat(number: widget.bookingData?.couponPrice)}",
                          style: Style.interRegular(size: 14))
                    ],
                  ),
                ],
              ),
            10.verticalSpace,
            Row(
              children: [
                Text(
                  AppHelpers.getTranslation(TrKeys.commissionFee),
                  style: Style.interNormal(size: 14),
                ),
                const Spacer(),
                Text(
                  AppHelpers.numberFormat(
                      number: widget.bookingData?.commissionFee),
                  style: Style.interRegular(size: 14),
                )
              ],
            ),
            10.verticalSpace,
            Row(
              children: [
                Text(
                  AppHelpers.getTranslation(TrKeys.totalPrice),
                  style: Style.interNormal(size: 14),
                ),
                const Spacer(),
                Text(
                  AppHelpers.numberFormat(
                      number: widget.bookingData?.totalPrice),
                  style: Style.interRegular(size: 14),
                )
              ],
            ),
            10.verticalSpace,
            CustomLine(),
            26.verticalSpace,
            Text(
              AppHelpers.getTranslation(TrKeys.thankYou).toUpperCase(),
              style: Style.interNormal(),
            ),
            24.verticalSpace,
            Row(
              mainAxisAlignment: MainAxisAlignment.spaceBetween,
              children: [
                Expanded(
                  child: LoginButton(
                      title: AppHelpers.getTranslation(TrKeys.print),
                      onPressed: () async {
                        if(LocalStorage.getPrinterData()==null){
                          final bytes = await _getBytes();
                          if (context.mounted) {
                            AppHelpers.showAlertDialog(
                                context: context, child: PrintPage(bytes: bytes));
                          }
                        }else {
                          final bytes = await _getBytes();
                          _printEscPos(
                            bytes,
                          );
                        }
                      }),
                ),
                10.horizontalSpace,
                CustomMiniButton(
                  size: 56,
                  onTap: () async {
                    final bytes = await _getBytes();
                    if (context.mounted) {
                      AppHelpers.showAlertDialog(
                          context: context, child: PrintPage(bytes: bytes));
                    }
                    // ignore: deprecated_member_use
                  },
                  icon: Remix.settings_5_fill,
                  iconColor: Style.white.withOpacity(0.7),
                  color: Style.primary,
                )
              ],
            ),
          ],
        ),
      ),
    );
  }

  var printerManager = PrinterManager.instance;

  List<int>? pendingTask;

  final _reconnect = false;

  final BTStatus _currentStatus = BTStatus.none;

  void _printEscPos(
    List<int> bytes,
  ) async {
    final profile = await CapabilityProfile.load();
    final generator = Generator(PaperSize.mm58, profile);
    var connectedTCP = false;
    if (LocalStorage.getPrinterData() == null) return;
    var bluetoothPrinter = LocalStorage.getPrinterData()!;

    switch (bluetoothPrinter.typePrinter) {
      case PrinterType.usb:
        debugPrint("");
        bytes += generator.feed(2);
        bytes += generator.cut();
        await printerManager.connect(
            type: bluetoothPrinter.typePrinter,
            model: UsbPrinterInput(
                name: bluetoothPrinter.deviceName,
                productId: bluetoothPrinter.productId,
                vendorId: bluetoothPrinter.vendorId));
        pendingTask = null;
        break;
      case PrinterType.bluetooth:
        bytes += generator.cut();
        await printerManager.connect(
            type: bluetoothPrinter.typePrinter,
            model: BluetoothPrinterInput(
                name: bluetoothPrinter.deviceName,
                address: bluetoothPrinter.address!,
                isBle: bluetoothPrinter.isBle ?? false,
                autoConnect: _reconnect));
        pendingTask = null;
        if (Platform.isAndroid) pendingTask = bytes;
        break;
      case PrinterType.network:
        bytes += generator.feed(2);
        bytes += generator.cut();
        connectedTCP = await printerManager.connect(
            type: bluetoothPrinter.typePrinter,
            model: TcpPrinterInput(ipAddress: bluetoothPrinter.address!));
        if (!connectedTCP) debugPrint(' --- please review your connection ---');
        break;
    }
    if (bluetoothPrinter.typePrinter == PrinterType.bluetooth &&
        Platform.isAndroid) {
      if (_currentStatus == BTStatus.connected) {
        printerManager.send(type: bluetoothPrinter.typePrinter, bytes: bytes);
        pendingTask = null;
      }
    } else {
      printerManager.send(type: bluetoothPrinter.typePrinter, bytes: bytes);
    }
  }

  Future<List<int>> _getBytes() async {
    List<int> bytes = [];
    String customDivider = "-------------------------------------------";
    String customLine = "___________________________________________";
    String customSpace = "  ";
    final profile = await CapabilityProfile.load(name: 'XP-N160I');
    //     // PaperSize.mm80 or PaperSize.mm58
    final generator = Generator(PaperSize.mm80, profile);

    bytes += generator.setGlobalCodeTable('CP1252');
    bytes += generator.setStyles(const PosStyles(align: PosAlign.center));
    bytes += generator.text(AppHelpers.getTranslation(TrKeys.bookingInvoice),
        styles: const PosStyles(align: PosAlign.center, bold: true));

    bytes += generator.text(
        "${AppHelpers.getTranslation(TrKeys.booking)} #${AppHelpers.getTranslation(TrKeys.id)}${widget.bookingData?.id}",
        styles: const PosStyles(align: PosAlign.center));
    bytes += generator.text(customDivider);
    bytes += generator.row([
      PosColumn(
          width: 7,
          text: customSpace + AppHelpers.getTranslation(TrKeys.shopName),
          styles: const PosStyles(align: PosAlign.left, codeTable: 'CP1252')),
      PosColumn(
          width: 5,
          text: LocalStorage.getShop()?.translation?.title ?? "",
          styles: const PosStyles(align: PosAlign.right, codeTable: 'CP1252')),
    ]);
    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.client),
        styles: const PosStyles(align: PosAlign.left),
      ),
      PosColumn(
        width: 6,
        text:
            "${widget.bookingData?.user?.firstname ?? ""} ${widget.bookingData?.user?.lastname ?? ""}",
        styles: const PosStyles(align: PosAlign.right),
      ),
    ]);
    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.master),
        styles: const PosStyles(
          align: PosAlign.left,
        ),
      ),
      PosColumn(
        width: 6,
        text:
            "${widget.bookingData?.master?.firstname ?? ""} ${widget.bookingData?.master?.lastname ?? ""}",
        styles: const PosStyles(
          align: PosAlign.right,
        ),
      ),
    ]);
    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.date),
        styles: const PosStyles(align: PosAlign.left),
      ),
      PosColumn(
        width: 6,
        text: TimeService.dateFormatMDYHm(
            widget.bookingData?.createdAt?.toLocal() ?? DateTime.now()),
        styles: const PosStyles(align: PosAlign.right),
      ),
    ]);

    bytes += generator.text(customLine);

    bytes += generator.setStyles(const PosStyles(align: PosAlign.center));

    bytes += generator.text(customDivider);
    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.service),
        styles: const PosStyles(align: PosAlign.left, bold: true),
      ),
      PosColumn(
        width: 6,
        text: widget.bookingData?.serviceMaster?.service?.translation?.title ??
            "",
        styles: const PosStyles(align: PosAlign.right, bold: true),
      ),
    ]);

    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.bookingDate),
        styles: const PosStyles(align: PosAlign.left, bold: true),
      ),
      PosColumn(
        width: 6,
        text: TimeService.dateFormatDMY(widget.bookingData?.startDate),
        styles: const PosStyles(align: PosAlign.right, bold: true),
      ),
    ]);

    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.bookingTime),
        styles: const PosStyles(align: PosAlign.left, bold: true),
      ),
      PosColumn(
        width: 6,
        text:
            "${TimeService.timeFormat(widget.bookingData?.startDate)}-${TimeService.timeFormat(widget.bookingData?.endDate)}",
        styles: const PosStyles(align: PosAlign.right, bold: true),
      ),
    ]);
    bytes += generator.text(customDivider);

    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.subtotal),
        styles: const PosStyles(align: PosAlign.left, bold: true),
      ),
      PosColumn(
        width: 6,
        text: AppHelpers.numberFormat(number: widget.bookingData?.price),
        styles: const PosStyles(align: PosAlign.right, bold: true),
      ),
    ]);
    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.commissionFee),
        styles: const PosStyles(align: PosAlign.left, bold: true),
      ),
      PosColumn(
        width: 6,
        text:
            AppHelpers.numberFormat(number: widget.bookingData?.commissionFee),
        styles: const PosStyles(align: PosAlign.right, bold: true),
      ),
    ]);

    if ((widget.bookingData?.discount ?? 0) != 0) {
      bytes += generator.row([
        PosColumn(
          width: 6,
          text: customSpace + AppHelpers.getTranslation(TrKeys.discount),
          styles: const PosStyles(align: PosAlign.left, bold: true),
        ),
        PosColumn(
          width: 6,
          text: AppHelpers.numberFormat(number: widget.bookingData?.discount),
          styles: const PosStyles(align: PosAlign.right, bold: true),
        ),
      ]);
    }
    bytes += generator.row([
      PosColumn(
        width: 6,
        text: customSpace + AppHelpers.getTranslation(TrKeys.totalPrice),
        styles: const PosStyles(align: PosAlign.left, bold: true),
      ),
      PosColumn(
        width: 6,
        text: AppHelpers.numberFormat(number: widget.bookingData?.totalPrice),
        styles: const PosStyles(align: PosAlign.right, bold: true),
      ),
    ]);
    bytes += generator.text(customDivider);
    bytes += generator.text(AppHelpers.getTranslation(TrKeys.thankYou));
    return bytes;
  }
}
